/*
 *   kioskrun.h
 *
 *   Copyright (C) 2004 Waldo Bastian <bastian@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2 as
 *   published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef _KIOSKRUN_H_
#define _KIOSKRUN_H_

#include <tqdict.h>
#include <tqobject.h>
#include <tqregexp.h>
#include <tqstringlist.h>
#include <tqtimer.h>

#include <dcopclient.h>
#include <dcopref.h>

#include <kprogress.h>

class ImmutableStatus;
class TDEConfig;
class TDEProcess;
class TDESimpleConfig;

class KioskGui;

class KioskRun : public TQObject
{
  friend class KioskGui;

  TQ_OBJECT
  
public:  
  static KioskRun* self() { return s_self; }

  void setKdeDirs(const TQStringList &dirs);
  void setUser(const TQString &user);

  TQString homeDir() { return m_homeDir; }
  TQStringList kdeDirs() { return m_kdeDirs; }
  TQString desktopPath() { return m_desktopPath; }

  // Locate existing anywhere
  TQString locate(const char *resource, const TQString &filename=TQString());
  
  // Locate for saving
  TQString locateSave(const char *resource, const TQString &filename=TQString());
  
  // Locate for reading saved changed
  TQString locateLocal(const char *resource, const TQString &filename=TQString());

  // Prepare runtime environment for run()
  bool prepare();
  
  // Update sycoca database in runtime environment
  void updateSycoca();

  // Request sycoca update in install environment after flushing config files
  void scheduleSycocaUpdate();

  // Request sycoca update in install environment
  void forceSycocaUpdate();

  // Run a program inside the runtime test environment
  TDEProcess* run(const TQString &cmd, const TQStringList &args=TQStringList());
  
  // A DCOPRef to make dcop calls into the runtime test environment
  DCOPRef dcopRef(const TQCString &appId, const TQCString &objId);
  
  // A DCOPClient to make dcop calls into the runtime test environment
  DCOPClient *dcopClient() { return m_dcopClient; }

  // Open config file in the install directory
  TDEConfig *configFile(const TQString &filename);

  // Make config files temporary mutable.
  void makeMutable(bool bMutable);

  // Returns whether specific config group is immutable, 
  // or entire file if group is empty
  bool isConfigImmutable(const TQString &filename, const TQString &group);
  
  // Make specific config group immutable,
  // or entire file if group is empty
  void setConfigImmutable(const TQString &filename, const TQString &group, bool bImmutable);

  // Close all opened config files.
  bool flushConfigCache();

  // Return all config files created by the user
  TQStringList newConfigFiles();

  // Merge new settings from the test directory into the installation directory
  void mergeConfigFile(const TQString &filename);

  // Lookup the setting for a custom action
  bool lookupCustomAction(const TQString &action);

  // Change the setting for a custom action
  void setCustomAction(const TQString &action, bool checked);

  // Create installation directory and its parent dirs
  bool createDir(const TQString &dir);

  // Install file
  bool install(const TQString &file, const TQString &destination);

  // Delete file
  bool remove(const TQString &destination);

  // Move file or directory
  bool move(const TQString &source, const TQString &destination, const TQStringList &files);

  // Delete directory in test home dir
  void deleteDir(const TQString &);

  // Open /etc/tderc for writing
  TDESimpleConfig *openKderc();
  
  // Install new /etc/tderc
  bool closeKderc();

  // Read information of profile @p profile
  void getProfileInfo(const TQString &profile, TQString &description, TQString &installDir, TQString &installUser);

  // Store information for profile @p profile
  bool setProfileInfo(const TQString &profile, const TQString &description, const TQString &installDir, const TQString &installUser, bool b=false, bool deleteFiles=true);

  // Get new, non-existing, profile name
  TQString newProfile();

  // Delete profile @p profile
  bool deleteProfile(const TQString &profile, bool deleteFiles = true);
  
  // Get list of all existing profiles
  TQStringList allProfiles();

  // Maps a single group or user to a one or more profiles
  typedef TQMap<TQString,TQStringList> ProfileMapping;

  // Read mappings between groups/users and profiles
  void getUserProfileMappings( ProfileMapping &groups, ProfileMapping &users, TQStringList &groupOrder);
  
  // Store mappings between groups/users and profiles
  bool setUserProfileMappings( const ProfileMapping &groups, const ProfileMapping &users, const TQStringList &groupOrder);

  // Read profile prefix
  TQString getProfilePrefix();
  
  // Store profile prefix
  bool setProfilePrefix(const TQString &prefix);

  // Create upload directory
  bool createRemoteDir(const KURL &dir);

  // Create upload directory and all its parent dirs and be polite if ask = true
  bool createRemoteDirRecursive(const KURL &dir, bool ask);

  // Upload file
  bool uploadRemote(const TQString &file, const KURL &dest);

protected:  
  KioskRun( TQObject* parent = 0, const char* name = 0);
  ~KioskRun();

  bool setupRuntimeEnv();
  void shutdownRuntimeEnv();
  void setupConfigEnv();
  void shutdownConfigEnv();
  void applyEnvironment(TDEProcess *p);

  TQString saveImmutableStatus(const TQString &filename);
  bool restoreImmutableStatus(const TQString &filename, bool force);

  void setCustomRestrictionFileBrowsing(bool restrict);

private:
  static KioskRun* s_self;
  TQString m_homeDir;
  TQString m_configDir;
  TQString m_desktopPath;
  TQString m_user;
  TQStringList m_kdeDirs;
  TQStringList m_xdgDataDirs;
  TQStringList m_xdgConfigDirs;
  DCOPClient *m_dcopClient;
  TDEInstance *m_instance;
  TDEInstance *m_saveInstance;
  TQDict<TDEConfig> m_saveConfigCache;
  TQDict<ImmutableStatus> m_immutableStatusCache;
  bool m_noRestrictions;
  bool m_forceSycocaUpdate;
  bool m_isRoot;
  
  TQString m_tderc;
  TQString m_localKderc;
  TDESimpleConfig *m_localKdercConfig;
};


class KioskRunProgressDialog : public KProgressDialog
{
   TQ_OBJECT
  
public:
   KioskRunProgressDialog(TQWidget *parent, const char *name, 
                          const TQString &caption, const TQString &text);
public slots:   
   void slotProgress();
   void slotFinished();

private:
   TQTimer m_timer;
   int m_timeStep;
};


#endif
