/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz <peter.penz@gmx.at>                  *
 *   Copyright (C) 2006 by Stefan Monov <logixoul@gmail.com>               *
 *   Copyright (C) 2006 by Cvetoslav Ludmiloff <ludmiloff@gmail.com>       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "dolphin.h"

#include <assert.h>

#include <kbookmarkmanager.h>
#include <tdeglobal.h>
#include <kpropertiesdialog.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <tdeversion.h>
#include <kstatusbar.h>
#include <tdeaccel.h>
#include <tdeio/netaccess.h>
#include <tdefiledialog.h>
#include <tdeconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <tdestdaccel.h>
#include <tdeaction.h>
#include <kstdaction.h>
#include <tdeactionclasses.h>
#include <tdepopupmenu.h>
#include <tdeio/renamedlg.h>
#include <kinputdialog.h>
#include <kshell.h>
#include <tdedesktopfile.h>
#include <tdestandarddirs.h>
#include <kprotocolinfo.h>
#include <tdemessagebox.h>
#include <kservice.h>
#include <tdeprocess.h>
#include <krun.h>

#include <tqclipboard.h>
#include <tqdragobject.h>

#include "urlnavigator.h"
#include "viewpropertiesdialog.h"
#include "viewproperties.h"
#include "dolphinsettings.h"
#include "dolphinsettingsdialog.h"
#include "dolphinstatusbar.h"
#include "undomanager.h"
#include "progressindicator.h"
#include "dolphinsettings.h"
#include "sidebars.h"
#include "sidebarssettings.h"


Dolphin& Dolphin::mainWin()
{
    static Dolphin* instance = 0;
    if (instance == 0) {
        instance = new Dolphin();
        instance->init();
    }
    return *instance;
}

Dolphin::~Dolphin()
{
}

void Dolphin::setActiveView(DolphinView* view)
{
    assert((view == m_view[PrimaryIdx]) || (view == m_view[SecondaryIdx]));
    if (m_activeView == view) {
        return;
    }

    m_activeView = view;

    updateHistory();
    updateEditActions();
    updateViewActions();
    updateGoActions();

    setCaption(m_activeView->url().fileName());

    emit activeViewChanged();
}

void Dolphin::dropURLs(const KURL::List& urls,
                       const KURL& destination)
{
    const ButtonState keyboardState = TDEApplication::keyboardMouseState();
    const bool shiftPressed = (keyboardState & ShiftButton) > 0;
    const bool controlPressed = (keyboardState & ControlButton) > 0;

    int selectedIndex = -1;
    if (shiftPressed && controlPressed) {
        // shortcut for 'Linke Here' is used
        selectedIndex = 2;
    }
    else if (controlPressed) {
        // shortcut for 'Copy Here' is used
        selectedIndex = 1;
    }
    else if (shiftPressed) {
        // shortcut for 'Move Here' is used
        selectedIndex = 0;
    }
    else {
        // no shortcut is used, hence open a popup menu
        TDEPopupMenu popup(this);

        popup.insertItem(SmallIcon("goto"), i18n("&Move Here") + "\t" + KKey::modFlagLabel(KKey::SHIFT), 0);
        popup.insertItem(SmallIcon("edit-copy"), i18n( "&Copy Here" ) + "\t" + KKey::modFlagLabel(KKey::CTRL), 1);
        popup.insertItem(i18n("&Link Here") + "\t" + KKey::modFlagLabel((KKey::ModFlag)(KKey::CTRL|KKey::SHIFT)), 2);
        popup.insertSeparator();
        popup.insertItem(SmallIcon("process-stop"), i18n("Cancel"), 3);
        popup.setAccel(i18n("Escape"), 3);

        selectedIndex = popup.exec(TQCursor::pos());
    }

    if (selectedIndex < 0) {
        return;
    }

    switch (selectedIndex) {
        case 0: {
            // 'Move Here' has been selected
            updateViewProperties(urls);
            moveURLs(urls, destination);
            break;
        }

        case 1: {
            // 'Copy Here' has been selected
            updateViewProperties(urls);
            copyURLs(urls, destination);
            break;
        }

        case 2: {
            // 'Link Here' has been selected
            TDEIO::Job* job = TDEIO::link(urls, destination);
            addPendingUndoJob(job, DolphinCommand::Link, urls, destination);
            break;
        }

        default:
            // 'Cancel' has been selected
            break;
    }
}

void Dolphin::refreshViews()
{
    const bool split = DolphinSettings::instance().isViewSplit();
    const bool isPrimaryViewActive = (m_activeView == m_view[PrimaryIdx]);
    DolphinSettings& settings = DolphinSettings::instance();
    KURL url;
    for (int i = PrimaryIdx; i <= SecondaryIdx; ++i) {
       if (m_view[i] != 0) {
            url = m_view[i]->url();

            // delete view instance...
            m_view[i]->close();
            m_view[i]->deleteLater();
            m_view[i] = 0;
        }

        if (split || (i == PrimaryIdx)) {
            // ... and recreate it
            ViewProperties props(url);
            m_view[i] = new DolphinView(m_splitter,
                                        url,
                                        props.viewMode(),
                                        props.isShowHiddenFilesEnabled());
            m_view[i]->show();
        }

        rightSidebarSettings* rightsidebarSettings = settings.rightsidebar();
        assert(rightsidebarSettings != 0);
        if (rightsidebarSettings->isVisible()) {
            m_splitter->moveToLast(m_rightsidebar);
        }
    }

    m_activeView = isPrimaryViewActive ? m_view[PrimaryIdx] : m_view[SecondaryIdx];
    assert(m_activeView != 0);

    updateViewActions();
    emit activeViewChanged();
}

void Dolphin::slotHistoryChanged()
{
    updateHistory();
}

void Dolphin::slotURLChanged(const KURL& url)
{
    updateEditActions();
    updateGoActions();
    setCaption(url.fileName());
}

void Dolphin::slotURLChangeRequest(const KURL& url)
{
	clearStatusBar();
	m_activeView->setURL(url);
}

void Dolphin::slotViewModeChanged()
{
    updateViewActions();
}

void Dolphin::slotShowHiddenFilesChanged()
{
    TDEToggleAction* showHiddenFilesAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_hidden_files"));
    showHiddenFilesAction->setChecked(m_activeView->isShowHiddenFilesEnabled());
}

void Dolphin::slotShowFilterBarChanged()
{
    TDEToggleAction* showFilterBarAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_filter_bar"));
    showFilterBarAction->setChecked(m_activeView->isFilterBarVisible());
}

void Dolphin::slotSortingChanged(DolphinView::Sorting sorting)
{
    TDEAction* action = 0;
    switch (sorting) {
        case DolphinView::SortByName:
            action = actionCollection()->action("by_name");
            break;
        case DolphinView::SortBySize:
            action = actionCollection()->action("by_size");
            break;
        case DolphinView::SortByDate:
            action = actionCollection()->action("by_date");
            break;
        default:
            break;
    }

    if (action != 0) {
        TDEToggleAction* toggleAction = static_cast<TDEToggleAction*>(action);
        toggleAction->setChecked(true);
    }
}

void Dolphin::slotSortOrderChanged(TQt::SortOrder order)
{
    TDEToggleAction* descending = static_cast<TDEToggleAction*>(actionCollection()->action("descending"));
    const bool sortDescending = (order == TQt::Descending);
    descending->setChecked(sortDescending);
}

void Dolphin::slotSelectionChanged()
{
    updateEditActions();

    assert(m_view[PrimaryIdx] != 0);
    int selectedURLsCount = m_view[PrimaryIdx]->selectedURLs().count();
    if (m_view[SecondaryIdx] != 0) {
        selectedURLsCount += m_view[SecondaryIdx]->selectedURLs().count();
    }

    TDEAction* compareFilesAction = actionCollection()->action("compare_files");
    compareFilesAction->setEnabled(selectedURLsCount == 2);

    m_activeView->updateStatusBar();

    emit selectionChanged();
}

void Dolphin::closeEvent(TQCloseEvent* event)
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("General");
    config->writeEntry("First Run", false);

    DolphinSettings& settings = DolphinSettings::instance();

    leftSidebarSettings* leftsidebarSettings = settings.leftsidebar();
    const bool isleftSidebarVisible = (m_leftsidebar != 0);
    leftsidebarSettings->setVisible(isleftSidebarVisible);
    if (isleftSidebarVisible) {
        leftsidebarSettings->setWidth(m_leftsidebar->width());
    }
    
    rightSidebarSettings* rightsidebarSettings = settings.rightsidebar();
    const bool isrightSidebarVisible = (m_rightsidebar != 0);
    rightsidebarSettings->setVisible(isrightSidebarVisible);
    if (isrightSidebarVisible) {
        rightsidebarSettings->setWidth(m_rightsidebar->width());
    }

    settings.save();

    config->sync();
    TDEMainWindow::closeEvent(event);
}

void Dolphin::saveProperties(TDEConfig* config)
{
    config->setGroup("Primary view");
    config->writeEntry("URL", m_view[PrimaryIdx]->url().url());
    config->writeEntry("Editable URL", m_view[PrimaryIdx]->isURLEditable());
    if (m_view[SecondaryIdx] != 0) {
        config->setGroup("Secondary view");
        config->writeEntry("URL", m_view[SecondaryIdx]->url().url());
        config->writeEntry("Editable URL", m_view[SecondaryIdx]->isURLEditable());
    }
}

void Dolphin::readProperties(TDEConfig* config)
{
    config->setGroup("Primary view");
    m_view[PrimaryIdx]->setURL(config->readEntry("URL"));
    m_view[PrimaryIdx]->setURLEditable(config->readBoolEntry("Editable URL"));
    if (config->hasGroup("Secondary view")) {
        config->setGroup("Secondary view");
        if (m_view[SecondaryIdx] == 0) {
            toggleSplitView();
        }
        m_view[SecondaryIdx]->setURL(config->readEntry("URL"));
        m_view[SecondaryIdx]->setURLEditable(config->readBoolEntry("Editable URL"));
    }
    else if (m_view[SecondaryIdx] != 0) {
        toggleSplitView();
    }
}

void Dolphin::createFolder()
{
    // Parts of the following code have been taken
    // from the class KonqPopupMenu located in
    // libqonq/konq_popupmenu.h of Konqueror.
    // (Copyright (C) 2000  David Faure <faure@kde.org>,
    // Copyright (C) 2001 Holger Freyther <freyther@yahoo.com>)

    clearStatusBar();

    DolphinStatusBar* statusBar = m_activeView->statusBar();
    const KURL baseURL(m_activeView->url());

    TQString name(i18n("New Folder"));
    if (baseURL.isLocalFile() && TQFileInfo(baseURL.path(+1) + name).exists()) {
        name = TDEIO::RenameDlg::suggestName(baseURL, i18n("New Folder"));
    }

    bool ok = false;
    name = KInputDialog::getText(i18n("New Folder"),
                                 i18n("Enter folder name:" ),
                                 name,
                                 &ok,
                                 this);

    if (!ok) {
        // the user has pressed 'Cancel'
        return;
    }

    assert(!name.isEmpty());

    KURL url;
    if ((name[0] == '/') || (name[0] == '~')) {
        url.setPath(KShell::tildeExpand(name));
    }
    else {
        name = TDEIO::encodeFileName(name);
        url = baseURL;
        url.addPath(name);
    }
    ok = TDEIO::NetAccess::mkdir(url, this);

    // TODO: provide message type hint
    if (ok) {
        statusBar->setMessage(i18n("Created folder %1.").arg(url.path()),
                              DolphinStatusBar::OperationCompleted);

        DolphinCommand command(DolphinCommand::CreateFolder, KURL::List(), url);
        UndoManager::instance().addCommand(command);
    }
    else {
        // Creating of the folder has been failed. Check whether the creating
        // has been failed because a folder with the same name exists...
        if (TDEIO::NetAccess::exists(url, true, this)) {
            statusBar->setMessage(i18n("A folder named %1 already exists.").arg(url.path()),
                                  DolphinStatusBar::Error);
        }
        else {
            statusBar->setMessage(i18n("Creating of folder %1 failed.").arg(url.path()),
                                  DolphinStatusBar::Error);
        }

    }
}

void Dolphin::createFile()
{
    // Parts of the following code have been taken
    // from the class KonqPopupMenu located in
    // libqonq/konq_popupmenu.h of Konqueror.
    // (Copyright (C) 2000  David Faure <faure@kde.org>,
    // Copyright (C) 2001 Holger Freyther <freyther@yahoo.com>)

    clearStatusBar();

    // TODO: const Entry& entry = m_createFileTemplates[TQString(sender->name())];
    // should be enough. Anyway: the implemantation of [] does a linear search internally too.
    KSortableValueList<CreateFileEntry, TQString>::ConstIterator it = m_createFileTemplates.begin();
    KSortableValueList<CreateFileEntry, TQString>::ConstIterator end = m_createFileTemplates.end();

    bool found = false;
    CreateFileEntry entry;
    while (!found && (it != end)) {
        if ((*it).index() == sender()->name()) {
            entry = (*it).value();
            found = true;
        }
        else {
            ++it;
        }
    }

    DolphinStatusBar* statusBar = m_activeView->statusBar();
    if (!found || !TQFile::exists(entry.templatePath)) {
        statusBar->setMessage(i18n("Could not create file."), DolphinStatusBar::Error);
       return;
    }

    // Get the source path of the template which should be copied.
    // The source path is part of the URL entry of the desktop file.
    const int pos = entry.templatePath.findRev('/');
    TQString sourcePath(entry.templatePath.left(pos + 1));
    sourcePath += TDEDesktopFile(entry.templatePath, true).readPathEntry("URL");

    TQString name(i18n(entry.name.ascii()));
    // Most entry names end with "..." (e. g. "HTML File..."), which is ok for
    // menus but no good choice for a new file name -> remove the dots...
    name.replace("...", TQString());

    // add the file extension to the name
    name.append(sourcePath.right(sourcePath.length() - sourcePath.findRev('.')));

    // Check whether a file with the current name already exists. If yes suggest automatically
    // a unique file name (e. g. "HTML File" will be replaced by "HTML File_1").
    const KURL viewURL(m_activeView->url());
    const bool fileExists = viewURL.isLocalFile() &&
                            TQFileInfo(viewURL.path(+1) + TDEIO::encodeFileName(name)).exists();
    if (fileExists) {
        name = TDEIO::RenameDlg::suggestName(viewURL, name);
    }

    // let the user change the suggested file name
    bool ok = false;
    name = KInputDialog::getText(entry.name,
                                 entry.comment,
                                 name,
                                 &ok,
                                 this);
    if (!ok) {
        // the user has pressed 'Cancel'
        return;
    }

    // before copying the template to the destination path check whether a file
    // with the given name already exists
    const TQString destPath(viewURL.prettyURL() + "/" + TDEIO::encodeFileName(name));
    const KURL destURL(destPath);
    if (TDEIO::NetAccess::exists(destURL, false, this)) {
        statusBar->setMessage(i18n("A file named %1 already exists.").arg(name),
                              DolphinStatusBar::Error);
        return;
    }

    // copy the template to the destination path
    const KURL sourceURL(sourcePath);
    TDEIO::CopyJob* job = TDEIO::copyAs(sourceURL, destURL);
    job->setDefaultPermissions(true);
    if (TDEIO::NetAccess::synchronousRun(job, this)) {
        statusBar->setMessage(i18n("Created file %1.").arg(name),
                              DolphinStatusBar::OperationCompleted);

        KURL::List list;
        list.append(sourceURL);
        DolphinCommand command(DolphinCommand::CreateFile, list, destURL);
        UndoManager::instance().addCommand(command);

    }
    else {
        statusBar->setMessage(i18n("Creating of file %1 failed.").arg(name),
                              DolphinStatusBar::Error);
    }
}

void Dolphin::rename()
{
    clearStatusBar();
    m_activeView->renameSelectedItems();
}

void Dolphin::moveToTrash()
{
    clearStatusBar();
    KURL::List selectedURLs = m_activeView->selectedURLs();
    TDEIO::Job* job = TDEIO::trash(selectedURLs);
    addPendingUndoJob(job, DolphinCommand::Trash, selectedURLs, m_activeView->url());
}

void Dolphin::deleteItems()
{
    clearStatusBar();

    KURL::List list = m_activeView->selectedURLs();
    const uint itemCount = list.count();
    assert(itemCount >= 1);

    TQString text;
    if (itemCount > 1) {
        text = i18n("Do you really want to delete the %1 selected items?").arg(itemCount);
    }
    else {
        const KURL& url = list.first();
        text = i18n("Do you really want to delete '%1'?").arg(url.fileName());
    }

    const bool del = KMessageBox::warningContinueCancel(this,
                                                        text,
                                                        TQString(),
                                                        KGuiItem(i18n("Delete"), SmallIcon("edit-delete"))
                                                       ) == KMessageBox::Continue;
    if (del) {
        TDEIO::Job* job = TDEIO::del(list);
        connect(job, TQ_SIGNAL(result(TDEIO::Job*)),
                this, TQ_SLOT(slotHandleJobError(TDEIO::Job*)));
        connect(job, TQ_SIGNAL(result(TDEIO::Job*)),
                this, TQ_SLOT(slotDeleteFileFinished(TDEIO::Job*)));
    }
}

void Dolphin::properties()
{
    const KFileItemList* sourceList = m_activeView->selectedItems();
    if (sourceList == 0) {
        return;
    }

    KFileItemList list;
    KFileItemListIterator it(*sourceList);
    KFileItem* item = 0;
    while ((item = it.current()) != 0) {
        list.append(item);
        ++it;
    }

    new KPropertiesDialog(list, this);
}

void Dolphin::quit()
{
    close();
}

void Dolphin::slotHandleJobError(TDEIO::Job* job)
{
    if (job->error() != 0) {
        m_activeView->statusBar()->setMessage(job->errorString(),
                                              DolphinStatusBar::Error);
    }
}

void Dolphin::slotDeleteFileFinished(TDEIO::Job* job)
{
    if (job->error() == 0) {
        m_activeView->statusBar()->setMessage(i18n("Delete operation completed."),
                                               DolphinStatusBar::OperationCompleted);

        // TODO: In opposite to the 'Move to Trash' operation in the class KFileIconView
        // no rearranging of the item position is done when a file has been deleted.
        // This is bypassed by reloading the view, but it might be worth to investigate
        // deeper for the root of this issue.
        m_activeView->reload();
    }
}

void Dolphin::slotUndoAvailable(bool available)
{
    TDEAction* undoAction = actionCollection()->action(KStdAction::stdName(KStdAction::Undo));
    if (undoAction != 0) {
        undoAction->setEnabled(available);
    }
}

void Dolphin::slotUndoTextChanged(const TQString& text)
{
    TDEAction* undoAction = actionCollection()->action(KStdAction::stdName(KStdAction::Undo));
    if (undoAction != 0) {
        undoAction->setText(text);
    }
}

void Dolphin::slotRedoAvailable(bool available)
{
    TDEAction* redoAction = actionCollection()->action(KStdAction::stdName(KStdAction::Redo));
    if (redoAction != 0) {
        redoAction->setEnabled(available);
    }
}

void Dolphin::slotRedoTextChanged(const TQString& text)
{
    TDEAction* redoAction = actionCollection()->action(KStdAction::stdName(KStdAction::Redo));
    if (redoAction != 0) {
        redoAction->setText(text);
    }
}

void Dolphin::cut()
{
    m_clipboardContainsCutData = true;
    TQDragObject* data = new KURLDrag(m_activeView->selectedURLs(),
                                     widget());
    TQApplication::clipboard()->setData(data);
}

void Dolphin::copy()
{
    m_clipboardContainsCutData = false;
    TQDragObject* data = new KURLDrag(m_activeView->selectedURLs(),
                                     widget());
    TQApplication::clipboard()->setData(data);
}

void Dolphin::paste()
{
    TQClipboard* clipboard = TQApplication::clipboard();
    TQMimeSource* data = clipboard->data();
    if (!KURLDrag::canDecode(data)) {
        return;
    }

    clearStatusBar();

    KURL::List sourceURLs;
    KURLDrag::decode(data, sourceURLs);

    // per default the pasting is done into the current URL of the view
    KURL destURL(m_activeView->url());

    // check whether the pasting should be done into a selected directory
    KURL::List selectedURLs = m_activeView->selectedURLs();
    if (selectedURLs.count() == 1) {
        const KFileItem fileItem(S_IFDIR,
                                 KFileItem::Unknown,
                                 selectedURLs.first(),
                                 true);
        if (fileItem.isDir()) {
            // only one item is selected which is a directory, hence paste
            // into this directory
            destURL = selectedURLs.first();
        }
    }


    updateViewProperties(sourceURLs);
    if (m_clipboardContainsCutData) {
        moveURLs(sourceURLs, destURL);
        m_clipboardContainsCutData = false;
        clipboard->clear();
    }
    else {
        copyURLs(sourceURLs, destURL);
    }
}

void Dolphin::updatePasteAction()
{
    TDEAction* pasteAction = actionCollection()->action(KStdAction::stdName(KStdAction::Paste));
    if (pasteAction == 0) {
        return;
    }

    TQString text(i18n("Paste"));
    TQClipboard* clipboard = TQApplication::clipboard();
    TQMimeSource* data = clipboard->data();
    if (KURLDrag::canDecode(data)) {
        pasteAction->setEnabled(true);

        KURL::List urls;
        KURLDrag::decode(data, urls);
        const int count = urls.count();
        if (count == 1) {
            pasteAction->setText(i18n("Paste 1 File"));
        }
        else {
            pasteAction->setText(i18n("Paste %1 Files").arg(count));
        }
    }
    else {
        pasteAction->setEnabled(false);
        pasteAction->setText(i18n("Paste"));
    }

    if (pasteAction->isEnabled()) {
        KURL::List urls = m_activeView->selectedURLs();
        const uint count = urls.count();
        if (count > 1) {
            // pasting should not be allowed when more than one file
            // is selected
            pasteAction->setEnabled(false);
        }
        else if (count == 1) {
            // Only one file is selected. Pasting is only allowed if this
            // file is a directory.
            const KFileItem fileItem(S_IFDIR,
                                     KFileItem::Unknown,
                                     urls.first(),
                                     true);
            pasteAction->setEnabled(fileItem.isDir());
        }
    }
}

void Dolphin::selectAll()
{
    clearStatusBar();
    m_activeView->selectAll();
}

void Dolphin::invertSelection()
{
    clearStatusBar();
    m_activeView->invertSelection();
}
void Dolphin::setIconsView()
{
    m_activeView->setMode(DolphinView::IconsView);
}

void Dolphin::setDetailsView()
{
    m_activeView->setMode(DolphinView::DetailsView);
}

void Dolphin::setPreviewsView()
{
    m_activeView->setMode(DolphinView::PreviewsView);
}

void Dolphin::sortByName()
{
    m_activeView->setSorting(DolphinView::SortByName);
}

void Dolphin::sortBySize()
{
    m_activeView->setSorting(DolphinView::SortBySize);
}

void Dolphin::sortByDate()
{
    m_activeView->setSorting(DolphinView::SortByDate);
}

void Dolphin::toggleSortOrder()
{
    const TQt::SortOrder order = (m_activeView->sortOrder() == TQt::Ascending) ?
                                TQt::Descending :
                                TQt::Ascending;
    m_activeView->setSortOrder(order);
}

void Dolphin::toggleSplitView()
{
    if (m_view[SecondaryIdx] == 0) {
        const int newWidth = (m_view[PrimaryIdx]->width() - m_splitter->handleWidth()) / 2;

        // create a secondary view
        m_view[SecondaryIdx] = new DolphinView(m_splitter,
                                               m_view[PrimaryIdx]->url(),
                                               m_view[PrimaryIdx]->mode(),
                                               m_view[PrimaryIdx]->isShowHiddenFilesEnabled());

        TQValueList<int> list = m_splitter->sizes();
        assert(!list.isEmpty());
        list.pop_back();
        list.append(newWidth);
        list.append(newWidth);
        m_splitter->setSizes(list);
        m_view[SecondaryIdx]->show();
        if(m_rightsidebar != 0){
            closerightSidebar();
            openrightSidebar();
        }
    }
    else {
        // remove secondary view
        if (m_activeView == m_view[PrimaryIdx]) {
            m_view[SecondaryIdx]->close();
            m_view[SecondaryIdx]->deleteLater();
            m_view[SecondaryIdx] = 0;
            setActiveView(m_view[PrimaryIdx]);
        }
        else {
            // The secondary view is active, hence from the users point of view
            // the content of the secondary view should be moved to the primary view.
            // From an implementation point of view it is more efficient to close
            // the primary view and exchange the internal pointers afterwards.
            m_view[PrimaryIdx]->close();
            m_view[PrimaryIdx]->deleteLater();
            m_view[PrimaryIdx] = m_view[SecondaryIdx];
            m_view[SecondaryIdx] = 0;
            setActiveView(m_view[PrimaryIdx]);
        }
    }
}

void Dolphin::reloadView()
{
    clearStatusBar();
    m_activeView->reload();
}

void Dolphin::stopLoading()
{
}

void Dolphin::showHiddenFiles()
{
    clearStatusBar();

    const TDEToggleAction* showHiddenFilesAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_hidden_files"));
    const bool show = showHiddenFilesAction->isChecked();
    m_activeView->setShowHiddenFilesEnabled(show);
}

void Dolphin::showFilterBar()
{
    const TDEToggleAction* showFilterBarAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_filter_bar"));
    const bool show = showFilterBarAction->isChecked();
    m_activeView->slotShowFilterBar(show);
}

void Dolphin::zoomIn()
{
    m_activeView->zoomIn();
    updateViewActions();
}

void Dolphin::zoomOut()
{
    m_activeView->zoomOut();
    updateViewActions();
}

void Dolphin::editLocation()
{
    clearStatusBar();
    m_activeView->editURL();
}

void Dolphin::browse()
{
    clearStatusBar();
    m_activeView->setURLEditable(false);
}

void Dolphin::adjustViewProperties()
{
    clearStatusBar();
    ViewPropertiesDialog dlg(m_activeView);
    dlg.exec();
}

void Dolphin::goBack()
{
    clearStatusBar();
    m_activeView->goBack();
}

void Dolphin::goForward()
{
    clearStatusBar();
    m_activeView->goForward();
}

void Dolphin::goUp()
{
    clearStatusBar();
    m_activeView->goUp();
}

void Dolphin::goHome()
{
    clearStatusBar();
    m_activeView->goHome();
}

void Dolphin::openTerminal()
{
    TDEConfig * config = tdeApp->config();
    config->setGroup("General");
    TQString term = config->readPathEntry("TerminalApplication", "konsole");

    TDEProcess proc;
    proc << term;
    proc.setWorkingDirectory(m_activeView->url().path());
    proc.start(TDEProcess::DontCare);
}

void Dolphin::findFile()
{
    KRun::run("kfind", m_activeView->url());
}

void Dolphin::compareFiles()
{
    // The method is only invoked if exactly 2 files have
    // been selected. The selected files may be:
    // - both in the primary view
    // - both in the secondary view
    // - one in the primary view and the other in the secondary
    //   view
    assert(m_view[PrimaryIdx] != 0);

    KURL urlA;
    KURL urlB;
    KURL::List urls = m_view[PrimaryIdx]->selectedURLs();

    switch (urls.count()) {
        case 0: {
            assert(m_view[SecondaryIdx] != 0);
            urls = m_view[SecondaryIdx]->selectedURLs();
            assert(urls.count() == 2);
            urlA = urls[0];
            urlB = urls[1];
            break;
        }

        case 1: {
            urlA = urls[0];
            assert(m_view[SecondaryIdx] != 0);
            urls = m_view[SecondaryIdx]->selectedURLs();
            assert(urls.count() == 1);
            urlB = urls[0];
            break;
        }

        case 2: {
            urlA = urls[0];
            urlB = urls[1];
            break;
        }

        default: {
            // may not happen: compareFiles may only get invoked if 2
            // files are selected
            assert(false);
        }
    }

    TQString command("kompare -c \"");
    command.append(urlA.prettyURL());
    command.append("\" \"");
    command.append(urlB.prettyURL());
    command.append('\"');
    KRun::runCommand(command, "Kompare", "kompare");

}

void Dolphin::editSettings()
{
    // TODO: make a static method for opening the settings dialog
    DolphinSettingsDialog dlg;
    dlg.exec();
}

void Dolphin::addUndoOperation(TDEIO::Job* job)
{
    if (job->error() != 0) {
        slotHandleJobError(job);
    }
    else {
        const int id = job->progressId();

        // set iterator to the executed command with the current id...
        TQValueList<UndoInfo>::Iterator it = m_pendingUndoJobs.begin();
        const TQValueList<UndoInfo>::Iterator end = m_pendingUndoJobs.end();
        bool found = false;
        while (!found && (it != end)) {
            if ((*it).id == id) {
                found = true;
            }
            else {
                ++it;
            }
        }

        if (found) {
            DolphinCommand command = (*it).command;
            if (command.type() == DolphinCommand::Trash) {
                // To be able to perform an undo for the 'Move to Trash' operation
                // all source URLs must be updated with the trash URL. E. g. when moving
                // a file "test.txt" and a second file "test.txt" to the trash,
                // then the filenames in the trash are "0-test.txt" and "1-test.txt".
                TQMap<TQString, TQString> metaData = job->metaData();
                KURL::List newSourceURLs;

                KURL::List sourceURLs = command.source();
                KURL::List::Iterator sourceIt = sourceURLs.begin();
                const KURL::List::Iterator sourceEnd = sourceURLs.end();

                while (sourceIt != sourceEnd) {
                    TQMap<TQString, TQString>::ConstIterator metaIt = metaData.find("trashURL-" + (*sourceIt).path());
                    if (metaIt != metaData.end()) {
                        newSourceURLs.append(KURL(metaIt.data()));
                    }
                    ++sourceIt;
                }
                command.setSource(newSourceURLs);
            }

            UndoManager::instance().addCommand(command);
            m_pendingUndoJobs.erase(it);

            DolphinStatusBar* statusBar = m_activeView->statusBar();
            switch (command.type()) {
                case DolphinCommand::Copy:
                    statusBar->setMessage(i18n("Copy operation completed."),
                                          DolphinStatusBar::OperationCompleted);
                    break;
                case DolphinCommand::Move:
                    statusBar->setMessage(i18n("Move operation completed."),
                                          DolphinStatusBar::OperationCompleted);
                    break;
                case DolphinCommand::Trash:
                    statusBar->setMessage(i18n("Move to trash operation completed."),
                                          DolphinStatusBar::OperationCompleted);
                    break;
                default:
                    break;
            }
        }
    }
}

Dolphin::Dolphin() :
    TDEMainWindow(0, "D3lphin"),
    m_splitter(0),
    m_leftsidebar(0),
    m_rightsidebar(0),
    m_activeView(0),
    m_clipboardContainsCutData(false)
{
    m_view[PrimaryIdx] = 0;
    m_view[SecondaryIdx] = 0;

    m_fileGroupActions.setAutoDelete(true);

    // TODO: the following members are not used yet. See documentation
    // of Dolphin::linkGroupActions() and Dolphin::linkToDeviceActions()
    // in the header file for details.
    //m_linkGroupActions.setAutoDelete(true);
    //m_linkToDeviceActions.setAutoDelete(true);
}

void Dolphin::init()
{
    // Check whether Dolphin runs the first time. If yes then
    // a proper default window size is given at the end of Dolphin::init().
    TDEConfig* config = tdeApp->config();
    config->setGroup("General");
    const bool firstRun = config->readBoolEntry("First Run", true);

    setAcceptDrops(true);

    m_splitter = new TQSplitter(this);

    DolphinSettings& settings = DolphinSettings::instance();

    KBookmarkManager* manager = settings.bookmarkManager();
    assert(manager != 0);
    KBookmarkGroup root = manager->root();
    if (root.first().isNull()) {
        root.addBookmark(manager, i18n("Home"), settings.homeURL(), "folder_home");
        root.addBookmark(manager, i18n("System"), KURL("system:/"), "computer");
        root.addBookmark(manager, i18n("Storage Media"), KURL("system:/media"), "blockdevice");
        root.addBookmark(manager, i18n("Network"), KURL("remote:/"), "network_local");
        root.addBookmark(manager, i18n("Users' Folders"), KURL("system:/users"), "folder_home2");
        root.addBookmark(manager, i18n("Root"), KURL("/"), "folder_red");
        root.addBookmark(manager, i18n("Trash"), KURL("trash:/"), "trashcan_full");
    }

    const KURL& homeURL = root.first().url();
    setCaption(homeURL.fileName());
    ViewProperties props(homeURL);
    m_view[PrimaryIdx] = new DolphinView(m_splitter,
                                         homeURL,
                                         props.viewMode(),
                                         props.isShowHiddenFilesEnabled());

    m_activeView = m_view[PrimaryIdx];

    setCentralWidget(m_splitter);

    // open sidebars
    leftSidebarSettings* leftsidebarSettings = settings.leftsidebar();
    assert(leftsidebarSettings != 0);
    if (leftsidebarSettings->isVisible()) {
        openleftSidebar();
    }
    
    rightSidebarSettings* rightsidebarSettings = settings.rightsidebar();
    assert(rightsidebarSettings != 0);
    if (rightsidebarSettings->isVisible()) {
        openrightSidebar();
    }

    setupActions();
    setupGUI(Keys|Save|Create|ToolBar);
    createGUI(0, false);

    stateChanged("new_file");
    setAutoSaveSettings();

    TQClipboard* clipboard = TQApplication::clipboard();
    connect(clipboard, TQ_SIGNAL(dataChanged()),
            this, TQ_SLOT(updatePasteAction()));
    updatePasteAction();
    updateGoActions();

    setupCreateNewMenuActions();

    loadSettings();

    if (firstRun) {
        // assure a proper default size if Dolphin runs the first time
        resize(640, 480);
    }
}

void Dolphin::loadSettings()
{
    DolphinSettings& settings = DolphinSettings::instance();

    TDEToggleAction* splitAction = static_cast<TDEToggleAction*>(actionCollection()->action("split_view"));
    if (settings.isViewSplit()) {
        splitAction->setChecked(true);
        toggleSplitView();
    }

    updateViewActions();
}

void Dolphin::setupActions()
{
    // setup 'File' menu
    TDEAction* createFolder = new TDEAction(i18n("Folder..."), "Ctrl+N",
                                        this, TQ_SLOT(createFolder()),
                                        actionCollection(), "create_folder");
    createFolder->setIcon("folder");

    new TDEAction(i18n("Rename"), KKey(Key_F2),
                this, TQ_SLOT(rename()),
                actionCollection(), "rename");

    TDEAction* moveToTrashAction = new TDEAction(i18n("Move to Trash"), KKey(Key_Delete),
                                             this, TQ_SLOT(moveToTrash()),
                                             actionCollection(), "move_to_trash");
    moveToTrashAction->setIcon("edittrash");

    TDEAction* deleteAction = new TDEAction(i18n("Delete"), "Shift+Delete",
                                        this, TQ_SLOT(deleteItems()),
                                        actionCollection(), "delete");
    deleteAction->setIcon("edit-delete");

    new TDEAction(i18n("Propert&ies"), "edit", "Alt+Return",
                     this, TQ_SLOT(properties()),
                     actionCollection(), "properties");

    KStdAction::quit(this, TQ_SLOT(quit()), actionCollection());

    // setup 'Edit' menu
    UndoManager& undoManager = UndoManager::instance();
    KStdAction::undo(&undoManager,
                     TQ_SLOT(undo()),
                     actionCollection());
    connect(&undoManager, TQ_SIGNAL(undoAvailable(bool)),
            this, TQ_SLOT(slotUndoAvailable(bool)));
    connect(&undoManager, TQ_SIGNAL(undoTextChanged(const TQString&)),
            this, TQ_SLOT(slotUndoTextChanged(const TQString&)));

    KStdAction::redo(&undoManager,
                     TQ_SLOT(redo()),
                     actionCollection());
    connect(&undoManager, TQ_SIGNAL(redoAvailable(bool)),
            this, TQ_SLOT(slotRedoAvailable(bool)));
    connect(&undoManager, TQ_SIGNAL(redoTextChanged(const TQString&)),
            this, TQ_SLOT(slotRedoTextChanged(const TQString&)));

    KStdAction::cut(this, TQ_SLOT(cut()), actionCollection());
    KStdAction::copy(this, TQ_SLOT(copy()), actionCollection());
    KStdAction::paste(this, TQ_SLOT(paste()), actionCollection());

    new TDEAction(i18n("Select All"), "Ctrl+A",
                this, TQ_SLOT(selectAll()),
                actionCollection(), "select_all");

    new TDEAction(i18n("Invert Selection"), "Ctrl+Shift+A",
                this, TQ_SLOT(invertSelection()),
                actionCollection(), "invert_selection");

    // setup 'View' menu
    KStdAction::zoomIn(this,
                       TQ_SLOT(zoomIn()),
                       actionCollection());

    KStdAction::zoomOut(this,
                        TQ_SLOT(zoomOut()),
                        actionCollection());

    TDERadioAction* iconsView = new TDERadioAction(i18n("Icons"), "Ctrl+1",
                                                this, TQ_SLOT(setIconsView()),
                                                actionCollection(), "icons");
    iconsView->setExclusiveGroup("view_mode");
    iconsView->setIcon("view_icon");

    TDERadioAction* detailsView = new TDERadioAction(i18n("Details"), "Ctrl+2",
                                                 this, TQ_SLOT(setDetailsView()),
                                                 actionCollection(), "details");
    detailsView->setExclusiveGroup("view_mode");
    detailsView->setIcon("view_text");

    TDERadioAction* previewsView = new TDERadioAction(i18n("Previews"), "Ctrl+3",
                                                   this, TQ_SLOT(setPreviewsView()),
                                                   actionCollection(), "previews");
    previewsView->setExclusiveGroup("view_mode");
    previewsView->setIcon("gvdirpart");

    TDERadioAction* sortByName = new TDERadioAction(i18n("By Name"), 0,
                                                this, TQ_SLOT(sortByName()),
                                                actionCollection(), "by_name");
    sortByName->setExclusiveGroup("sort");

    TDERadioAction* sortBySize = new TDERadioAction(i18n("By Size"), 0,
                                                this, TQ_SLOT(sortBySize()),
                                                actionCollection(), "by_size");
    sortBySize->setExclusiveGroup("sort");

    TDERadioAction* sortByDate = new TDERadioAction(i18n("By Date"), 0,
                                                this, TQ_SLOT(sortByDate()),
                                                actionCollection(), "by_date");
    sortByDate->setExclusiveGroup("sort");

    new TDEToggleAction(i18n("Descending"), 0, this, TQ_SLOT(toggleSortOrder()),
                      actionCollection(), "descending");

    new TDEToggleAction(i18n("Show Hidden Files"), "Alt+.",
                      this, TQ_SLOT(showHiddenFiles()),
                      actionCollection(), "show_hidden_files");

    TDEToggleAction* splitAction = new TDEToggleAction(i18n("Split View"), "F10",
                                                   this, TQ_SLOT(toggleSplitView()),
                                                   actionCollection(), "split_view");
    splitAction->setIcon("view_left_right");

    TDEAction* reloadAction = new TDEAction(i18n("Reload"), "F5",
                                        this, TQ_SLOT(reloadView()),
                                        actionCollection(), "reload");
    reloadAction->setIcon("reload");

    TDEAction* stopAction = new TDEAction(i18n("Stop"), 0,
                                      this, TQ_SLOT(stopLoading()),
                                      actionCollection(), "stop");
    stopAction->setIcon("process-stop");

    new TDEAction(i18n("Edit Location"), "Ctrl+L",
                this, TQ_SLOT(editLocation()),
                actionCollection(), "edit_location");

    new TDEAction(i18n("Browse"), "Ctrl+B",
                this, TQ_SLOT(browse()),
                actionCollection(), "browse");

    new TDEToggleAction(i18n("Left Sidebar"), "F8",
                      this, TQ_SLOT(toggleleftSidebar()),
                      actionCollection(), "leftsidebar");
                      
    new TDEToggleAction(i18n("Right Sidebar"), "F9",
                      this, TQ_SLOT(togglerightSidebar()),
                      actionCollection(), "rightsidebar");

    new TDEAction(i18n("Adjust View Properties..."), 0,
                this, TQ_SLOT(adjustViewProperties()),
                actionCollection(), "view_properties");

    // setup 'Go' menu
    KStdAction::back(this, TQ_SLOT(goBack()), actionCollection());
    KStdAction::forward(this, TQ_SLOT(goForward()), actionCollection());
    KStdAction::up(this, TQ_SLOT(goUp()), actionCollection());
    KStdAction::home(this, TQ_SLOT(goHome()), actionCollection());

    // setup 'Tools' menu
    TDEAction* openTerminalAction = new TDEAction(i18n("Open Terminal"), "F4",
                                         this, TQ_SLOT(openTerminal()),
                                         actionCollection(), "open_terminal");
    openTerminalAction->setIcon("konsole");

    TDEAction* findFileAction = new TDEAction(i18n("Find File..."), "Ctrl+F",
                                         this, TQ_SLOT(findFile()),
                                         actionCollection(), "find_file");
    findFileAction->setIcon("filefind");

    new TDEToggleAction(i18n("Show Filter Bar"), "filter", "/",
                      this, TQ_SLOT(showFilterBar()),
                      actionCollection(), "show_filter_bar");

    TDEAction* compareFilesAction = new TDEAction(i18n("Compare Files"), 0,
                                          this, TQ_SLOT(compareFiles()),
                                          actionCollection(), "compare_files");
    compareFilesAction->setIcon("kompare");
    compareFilesAction->setEnabled(false);

    // setup 'Settings' menu
    KStdAction::preferences(this, TQ_SLOT(editSettings()), actionCollection());
}

void Dolphin::setupCreateNewMenuActions()
{
    // Parts of the following code have been taken
    // from the class KNewMenu located in
    // libqonq/knewmenu.h of Konqueror.
    //  Copyright (C) 1998, 1999 David Faure <faure@kde.org>
    //                2003       Sven Leiber <s.leiber@web.de>

    TQStringList files = actionCollection()->instance()->dirs()->findAllResources("templates");
    for (TQStringList::Iterator it = files.begin() ; it != files.end(); ++it) {
        if ((*it)[0] != '.' ) {
            TDESimpleConfig config(*it, true);
            config.setDesktopGroup();

            // tricky solution to ensure that TextFile is at the beginning
            // because this filetype is the most used (according kde-core discussion)
            const TQString name(config.readEntry("Name"));
            TQString key(name);

            const TQString path(config.readPathEntry("URL"));
            if (!path.endsWith("emptydir")) {
                if (path.endsWith("TextFile.txt")) {
                    key = "1" + key;
                }
                else if (!TDEDesktopFile::isDesktopFile(path)) {
                    key = "2" + key;
                }
                else if (path.endsWith("URL.desktop")){
                    key = "3" + key;
                }
                else if (path.endsWith("Program.desktop")){
                    key = "4" + key;
                }
                else {
                    key = "5";
                }

                const TQString icon(config.readEntry("Icon"));
                const TQString comment(config.readEntry("Comment"));
                const TQString type(config.readEntry("Type"));

                const TQString filePath(*it);


                if (type == "Link") {
                    CreateFileEntry entry;
                    entry.name = name;
                    entry.icon = icon;
                    entry.comment = comment;
                    entry.templatePath = filePath;
                    m_createFileTemplates.insert(key, entry);
                }
            }
        }
    }
    m_createFileTemplates.sort();

    unplugActionList("create_actions");
    KSortableValueList<CreateFileEntry, TQString>::ConstIterator it = m_createFileTemplates.begin();
    KSortableValueList<CreateFileEntry, TQString>::ConstIterator end = m_createFileTemplates.end();
    while (it != end) {
        CreateFileEntry entry = (*it).value();
        TDEAction* action = new TDEAction(entry.name);
        action->setIcon(entry.icon);
        action->setName((*it).index().ascii());
        connect(action, TQ_SIGNAL(activated()),
                this, TQ_SLOT(createFile()));

        const TQChar section = ((*it).index()[0]);
        switch (section) {
            case '1':
            case '2': {
                m_fileGroupActions.append(action);
                break;
            }

            case '3':
            case '4': {
                // TODO: not used yet. See documentation of Dolphin::linkGroupActions()
                // and Dolphin::linkToDeviceActions() in the header file for details.
                //m_linkGroupActions.append(action);
                break;
            }

            case '5': {
                // TODO: not used yet. See documentation of Dolphin::linkGroupActions()
                // and Dolphin::linkToDeviceActions() in the header file for details.
                //m_linkToDeviceActions.append(action);
                break;
            }
            default:
                break;
        }
        ++it;
    }

    plugActionList("create_file_group", m_fileGroupActions);
    //plugActionList("create_link_group", m_linkGroupActions);
    //plugActionList("link_to_device", m_linkToDeviceActions);
}

void Dolphin::updateHistory()
{
    int index = 0;
    const TQValueList<URLNavigator::HistoryElem> list = m_activeView->urlHistory(index);

    TDEAction* backAction = actionCollection()->action("go_back");
    if (backAction != 0) {
        backAction->setEnabled(index < static_cast<int>(list.count()) - 1);
    }

    TDEAction* forwardAction = actionCollection()->action("go_forward");
    if (forwardAction != 0) {
        forwardAction->setEnabled(index > 0);
    }
}

void Dolphin::updateEditActions()
{
    const KFileItemList* list = m_activeView->selectedItems();
    if ((list == 0) || (*list).isEmpty()) {
        stateChanged("has_no_selection");
    }
    else {
        stateChanged("has_selection");

        TDEAction* renameAction = actionCollection()->action("rename");
        if (renameAction != 0) {
            renameAction->setEnabled(list->count() >= 1);
        }

        bool enableMoveToTrash = true;

        KFileItemListIterator it(*list);
        KFileItem* item = 0;
        while ((item = it.current()) != 0) {
            const KURL& url = item->url();
            // only enable the 'Move to Trash' action for local files
            if (!url.isLocalFile()) {
                enableMoveToTrash = false;
            }
            ++it;
        }

        TDEAction* moveToTrashAction = actionCollection()->action("move_to_trash");
        moveToTrashAction->setEnabled(enableMoveToTrash);
    }
    updatePasteAction();
}

void Dolphin::updateViewActions()
{
    TDEAction* zoomInAction = actionCollection()->action(KStdAction::stdName(KStdAction::ZoomIn));
    if (zoomInAction != 0) {
        zoomInAction->setEnabled(m_activeView->isZoomInPossible());
    }

    TDEAction* zoomOutAction = actionCollection()->action(KStdAction::stdName(KStdAction::ZoomOut));
    if (zoomOutAction != 0) {
        zoomOutAction->setEnabled(m_activeView->isZoomOutPossible());
    }

    TDEAction* action = 0;
    switch (m_activeView->mode()) {
        case DolphinView::IconsView:
            action = actionCollection()->action("icons");
            break;
        case DolphinView::DetailsView:
            action = actionCollection()->action("details");
            break;
        case DolphinView::PreviewsView:
            action = actionCollection()->action("previews");
            break;
        default:
            break;
    }

    if (action != 0) {
        TDEToggleAction* toggleAction = static_cast<TDEToggleAction*>(action);
        toggleAction->setChecked(true);
    }

    slotSortingChanged(m_activeView->sorting());
    slotSortOrderChanged(m_activeView->sortOrder());

    TDEToggleAction* showFilterBarAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_filter_bar"));
    showFilterBarAction->setChecked(m_activeView->isFilterBarVisible());

    TDEToggleAction* showHiddenFilesAction =
        static_cast<TDEToggleAction*>(actionCollection()->action("show_hidden_files"));
    showHiddenFilesAction->setChecked(m_activeView->isShowHiddenFilesEnabled());

    TDEToggleAction* splitAction = static_cast<TDEToggleAction*>(actionCollection()->action("split_view"));
    splitAction->setChecked(m_view[SecondaryIdx] != 0);

    TDEToggleAction* leftsidebarAction = static_cast<TDEToggleAction*>(actionCollection()->action("leftsidebar"));
    leftsidebarAction->setChecked(m_leftsidebar != 0);
    
    TDEToggleAction* rightsidebarAction = static_cast<TDEToggleAction*>(actionCollection()->action("rightsidebar"));
    rightsidebarAction->setChecked(m_rightsidebar != 0);
}

void Dolphin::updateGoActions()
{
    TDEAction* goUpAction = actionCollection()->action(KStdAction::stdName(KStdAction::Up));
    const KURL& currentURL = m_activeView->url();
    goUpAction->setEnabled(currentURL.upURL() != currentURL);
}

void Dolphin::updateViewProperties(const KURL::List& urls)
{
    if (urls.isEmpty()) {
        return;
    }

    // Updating the view properties might take up to several seconds
    // when dragging several thousand URLs. Writing a TDEIO slave for this
    // use case is not worth the effort, but at least the main widget
    // must be disabled and a progress should be shown.
    ProgressIndicator progressIndicator(i18n("Updating view properties..."),
                                        TQString(),
                                        urls.count());

    KURL::List::ConstIterator end = urls.end();
    for(KURL::List::ConstIterator it = urls.begin(); it != end; ++it) {
        progressIndicator.execOperation();

        ViewProperties props(*it);
        props.save();
    }
}

void Dolphin::copyURLs(const KURL::List& source, const KURL& dest)
{
    TDEIO::Job* job = TDEIO::copy(source, dest);
    addPendingUndoJob(job, DolphinCommand::Copy, source, dest);
}

void Dolphin::moveURLs(const KURL::List& source, const KURL& dest)
{
    TDEIO::Job* job = TDEIO::move(source, dest);
    addPendingUndoJob(job, DolphinCommand::Move, source, dest);
}

void Dolphin::addPendingUndoJob(TDEIO::Job* job,
                                DolphinCommand::Type commandType,
                                const KURL::List& source,
                                const KURL& dest)
{
    connect(job, TQ_SIGNAL(result(TDEIO::Job*)),
            this, TQ_SLOT(addUndoOperation(TDEIO::Job*)));

    UndoInfo undoInfo;
    undoInfo.id = job->progressId();
    undoInfo.command = DolphinCommand(commandType, source, dest);
    m_pendingUndoJobs.append(undoInfo);
}

void Dolphin::clearStatusBar()
{
    m_activeView->statusBar()->clear();
}

void Dolphin::openleftSidebar()
{
    if (m_leftsidebar != 0) {
        // the sidebar is already open
        return;
    }

    m_leftsidebar = new leftSidebar(m_splitter);
    m_leftsidebar->show();

    connect(m_leftsidebar, TQ_SIGNAL(urlChanged(const KURL&)),
            this, TQ_SLOT(slotURLChangeRequest(const KURL&)));
    m_splitter->setCollapsible(m_leftsidebar, false);
    m_splitter->setResizeMode(m_leftsidebar, TQSplitter::KeepSize);
    m_splitter->moveToFirst(m_leftsidebar);

    leftSidebarSettings* settings = DolphinSettings::instance().leftsidebar();
    settings->setVisible(true);
}

void Dolphin::openrightSidebar()
{
    if (m_rightsidebar != 0) {
        // the sidebar is already open
        return;
    }

    m_rightsidebar = new rightSidebar(m_splitter);
    m_rightsidebar->show();

    connect(m_rightsidebar, TQ_SIGNAL(urlChanged(const KURL&)),
            this, TQ_SLOT(slotURLChangeRequest(const KURL&)));
    m_splitter->setCollapsible(m_rightsidebar, false);
    m_splitter->setResizeMode(m_rightsidebar, TQSplitter::KeepSize);
    m_splitter->moveToLast(m_rightsidebar);

    rightSidebarSettings* settings = DolphinSettings::instance().rightsidebar();
    settings->setVisible(true);
}

void Dolphin::closeleftSidebar()
{
    if (m_leftsidebar == 0) {
        // the sidebar has already been closed
        return;
    }

    // store width of sidebar and remember that the sidebar has been closed
    leftSidebarSettings* settings = DolphinSettings::instance().leftsidebar();
    settings->setVisible(false);
    settings->setWidth(m_leftsidebar->width());

    m_leftsidebar->deleteLater();
    m_leftsidebar = 0;
}

void Dolphin::closerightSidebar()
{
    if (m_rightsidebar == 0) {
        // the sidebar has already been closed
        return;
    }

    // store width of sidebar and remember that the sidebar has been closed
    rightSidebarSettings* settings = DolphinSettings::instance().rightsidebar();
    settings->setVisible(false);
    settings->setWidth(m_rightsidebar->width());

    m_rightsidebar->deleteLater();
    m_rightsidebar = 0;
}

void Dolphin::toggleleftSidebar()
{
    if (m_leftsidebar == 0) {
        openleftSidebar();
    }
    else {
        closeleftSidebar();
    }

    TDEToggleAction* leftsidebarAction = static_cast<TDEToggleAction*>(actionCollection()->action("leftsidebar"));
    leftsidebarAction->setChecked(m_leftsidebar != 0);
}

void Dolphin::togglerightSidebar()
{
    if (m_rightsidebar == 0) {
        openrightSidebar();
    }
    else {
        closerightSidebar();
    }

    TDEToggleAction* rightsidebarAction = static_cast<TDEToggleAction*>(actionCollection()->action("rightsidebar"));
    rightsidebarAction->setChecked(m_rightsidebar != 0);
}

#include "dolphin.moc"
